import React, { useEffect, useState } from "react";
import profilePic from "../../../../../../public/assets/profile/Avatar.png";
import Image from "next/image";
import { GetRequest, PostRequest } from "@/app/axios/axiosRequests";
import { useFormik } from "formik";
import * as Yup from "yup";
import toast from "react-hot-toast";
import { useGetUserDetailQuery } from "@/services/Api";
import { useSelector } from "react-redux";
import PhoneInput from "react-phone-input-2";
import "react-phone-input-2/lib/style.css";
const UserProfile = () => {
  const [data, setData] = useState(null);
  const userId = localStorage.getItem("userId");
  const globalProfileImg = useSelector((state) => state.auth.globalProfileImg);
  const [loader, setLoader] = useState(true);

  const { data: getUserDetail, refetch: getUserDetailRefetch } =
    useGetUserDetailQuery({ id: userId });

  useEffect(() => {
    if (getUserDetail === undefined) {
      setLoader(true);
    } else {
      setLoader(false);
    }

    if (getUserDetail) {
      setData(getUserDetail?.user);
    }
  }, [getUserDetail]);

  useEffect(() => {
    getUserDetailRefetch();
  }, []);

  const formik = useFormik({
    enableReinitialize: true,
    initialValues: {
      first_name: data?.first_name || "",
      last_name: data?.last_name || "",
      email: data?.email || "",
      phone: data?.phone || "",
      ssn: data?.ssn || "",
      password: "",
    },
    validationSchema: Yup.object({
      first_name: Yup.string().required("First Name is required"),
      last_name: Yup.string().required("Last Name is required"),
      email: Yup.string()
        .email("Invalid email address")
        .required("Email is required"),
      phone: Yup.string()
        .required("Phone number is required")
        .matches(/^\d{10,}$/, "The phone number is not correct"),
      ssn: Yup.string()
        .nullable()
        .matches(
          /^\d{3}-\d{2}-\d{4}$/,
          "SSN must be in the format 123-45-6789"
        ),

      password: Yup.string(),
    }),
    onSubmit: async (values, { setSubmitting }) => {
      try {
        console.log("Submitting values:", values);
        const userId = localStorage.getItem("userId");
        console.log("User ID:", userId);

        const response = await PostRequest(`/admin/users/${userId}`, values);
        console.log("Response:", response);

        toast.success("User information updated successfully", {
          position: "top-center",
        });
      } catch (error) {
        console.error(
          "Error updating user information:",
          error.response?.data || error.message
        ); // Log the error details
        toast.error("User information not updated", {
          position: "top-center",
        });
      } finally {
        setSubmitting(false);
      }
    },
  });

  return (
    <div className="">
      <div className="bg-[#FFFFFF] p-4 md:p-6 rounded-2xl border">
        <h2 class="mb-4 text-xl md:text-2xl font-bold">User Profile</h2>
        {loader ? (
          <div className="md:h-[354px] h-[250px] flex items-center justify-center p-4 rounded-2xl mt-4">
            <div class="flex justify-center items-center bg-white">
              <div class="border-t-4 border-[#0689BE] border-solid w-16 h-16 rounded-full animate-spin"></div>
            </div>
          </div>
        ) : (
          <form
            className="w-full flex flex-wrap"
            onSubmit={formik.handleSubmit}
          >
            <div className="flex flex-col md:flex-row gap-5 justify-center grow">
              <div className="flex flex-col items-center rounded-lg gap-5">
                <label class="form-label pt-1">User profile image</label>
                <img
                  src={getUserDetail.user.profile_photo_path ? `https://tracking.logoinnovators.com/backend/public/${getUserDetail?.user?.profile_photo_path}` : "https://m.media-amazon.com/images/I/5170BeVp0kL._AC_UL800_QL65_.jpg" }
                  alt="Profile Picture"
                  className="h-[100px] w-[100px] object-cover rounded-full"
                />
              </div>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 w-full justify-between gap-x-6 user-profile-form">
              <div className="mb-3">
                <label className="form-label">First Name</label>
                <input
                  type="text"
                  className="form-control w-full"
                  {...formik.getFieldProps("first_name")}
                />
                {formik.touched.first_name && formik.errors.first_name ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.first_name}
                  </div>
                ) : null}
              </div>
              <div className="mb-3">
                <label className="form-label">Last Name</label>
                <input
                  type="text"
                  className="w-full form-control"
                  {...formik.getFieldProps("last_name")}
                />
                {formik.touched.last_name && formik.errors.last_name ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.last_name}
                  </div>
                ) : null}
              </div>
              <div className="mb-3">
                <label className="form-label">Email</label>
                <input
                  type="email"
                  className="w-full form-control"
                  {...formik.getFieldProps("email")}
                />
                {formik.touched.email && formik.errors.email ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.email}
                  </div>
                ) : null}
              </div>
              <div className="mb-3">
                <label className="form-label">Phone </label>
                {/* <input
                type="text"
                className="bg-[#F6F9FB] w-full px-2 py-2 text-[14px] font-normal outline-none rounded-lg"
                {...formik.getFieldProps("phone")}
              /> */}
                <PhoneInput
                  country={"us"}
                  value={formik.values.phone || "+1"} // Ensure the formik value is passed here
                  onChange={(value) => {
                    formik.setFieldValue("phone", value); // Update formik value with phone input
                  }}
                  inputClass="w-full admin-phone-field-signup"
                  countryCodeEditable={false}
                />
                {formik.touched.phone && formik.errors.phone ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.phone}
                  </div>
                ) : null}
              </div>
              <div className="mb-3">
                <label className="form-label">SSN</label>
                <input
                  type="text"
                  className="w-full form-control"
                  {...formik.getFieldProps("ssn")}
                />
                {formik.touched.ssn && formik.errors.ssn ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.ssn}
                  </div>
                ) : null}
              </div>
              <div className="mb-3">
                <label className="form-label">Password</label>
                <input
                  type="password"
                  className="w-full form-control"
                  {...formik.getFieldProps("password")}
                />
                {formik.touched.password && formik.errors.password ? (
                  <div className="text-red-500 text-sm">
                    {formik.errors.password}
                  </div>
                ) : null}
              </div>
            </div>
            <div className="my-6 flex justify-end w-full">
              <button
                type="submit"
                className="px-8 py-3 text-[14px] text-white bg-[#0689BE] rounded-lg"
              >
                Update
              </button>
            </div>
          </form>
        )}
      </div>
    </div>
  );
};

export default UserProfile;
