import { NextResponse } from "next/server";
import { cookies } from "next/headers";

export function middleware(request) {
  const restrictedRoutes = [
    "/login",
    "/sign-up",
    "/forgot-password",
    "/reset-password",
  ];
  const userRoutes = [
    "/flights",
    "/profile",
    "/subscription",
    "/flights/detail",
  ];
  const adminRoutes = [
    "/users",
    "/billingsubscription",
    "/billingsubscription/:path*",
  ];

  const { pathname } = request.nextUrl;
  const authToken = cookies().get("authToken")?.value;
  const role = cookies().get("role")?.value;

  // Redirect if the user doesn't have authToken and tries to access userRoutes or adminRoutes
  if (!authToken || !role) {
    if (
      userRoutes.some((route) => pathname.startsWith(route)) ||
      adminRoutes.some((route) => pathname.startsWith(route))
    ) {
      const redirectUrl = new URL("/login", request.url);
      return NextResponse.redirect(redirectUrl);
    }
  }

  // Restrict access based on role
  if (
    role === "admin" &&
    userRoutes.some((route) => pathname.startsWith(route))
  ) {
    const redirectUrl = new URL("/", request.url);
    return NextResponse.redirect(redirectUrl);
  }

  if (
    role === "user" &&
    adminRoutes.some((route) => pathname.startsWith(route))
  ) {
    const redirectUrl = new URL("/", request.url);
    return NextResponse.redirect(redirectUrl);
  }

  // Redirect authenticated users away from restricted routes
  if (authToken && restrictedRoutes.includes(pathname)) {
    const redirectUrl = new URL("/", request.url);
    return NextResponse.redirect(redirectUrl);
  }

  // Allow unauthenticated users to access restricted routes
  if (!authToken && restrictedRoutes.includes(pathname)) {
    return NextResponse.next();
  }

  return NextResponse.next();
}

export const config = {
  matcher: ["/:path*"],
};
